﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/connect/model/Reference.h>
#include <aws/connect/model/SegmentAttributeValue.h>
#include <aws/connect/model/TaskAttachment.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Connect {
namespace Model {

/**
 */
class StartTaskContactRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API StartTaskContactRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartTaskContact"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the Amazon Connect instance. You can <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find
   * the instance ID</a> in the Amazon Resource Name (ARN) of the instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  StartTaskContactRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the previous chat, voice, or task contact. Any updates to
   * user-defined attributes to task contacts linked using the same
   * <code>PreviousContactID</code> will affect every contact in the chain. There can
   * be a maximum of 12 linked task contacts in a chain.</p>
   */
  inline const Aws::String& GetPreviousContactId() const { return m_previousContactId; }
  inline bool PreviousContactIdHasBeenSet() const { return m_previousContactIdHasBeenSet; }
  template <typename PreviousContactIdT = Aws::String>
  void SetPreviousContactId(PreviousContactIdT&& value) {
    m_previousContactIdHasBeenSet = true;
    m_previousContactId = std::forward<PreviousContactIdT>(value);
  }
  template <typename PreviousContactIdT = Aws::String>
  StartTaskContactRequest& WithPreviousContactId(PreviousContactIdT&& value) {
    SetPreviousContactId(std::forward<PreviousContactIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the flow for initiating the tasks. To see the ContactFlowId
   * in the Amazon Connect admin website, on the navigation menu go to
   * <b>Routing</b>, <b>Flows</b>. Choose the flow. On the flow page, under the name
   * of the flow, choose <b>Show additional flow information</b>. The ContactFlowId
   * is the last part of the ARN, shown here in bold: </p>
   * <p>arn:aws:connect:us-west-2:xxxxxxxxxxxx:instance/xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx/contact-flow/<b>846ec553-a005-41c0-8341-xxxxxxxxxxxx</b>
   * </p>
   */
  inline const Aws::String& GetContactFlowId() const { return m_contactFlowId; }
  inline bool ContactFlowIdHasBeenSet() const { return m_contactFlowIdHasBeenSet; }
  template <typename ContactFlowIdT = Aws::String>
  void SetContactFlowId(ContactFlowIdT&& value) {
    m_contactFlowIdHasBeenSet = true;
    m_contactFlowId = std::forward<ContactFlowIdT>(value);
  }
  template <typename ContactFlowIdT = Aws::String>
  StartTaskContactRequest& WithContactFlowId(ContactFlowIdT&& value) {
    SetContactFlowId(std::forward<ContactFlowIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A custom key-value pair using an attribute map. The attributes are standard
   * Amazon Connect attributes, and can be accessed in flows just like any other
   * contact attributes.</p> <p>There can be up to 32,768 UTF-8 bytes across all
   * key-value pairs per contact. Attribute keys can include only alphanumeric, dash,
   * and underscore characters.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAttributes() const { return m_attributes; }
  inline bool AttributesHasBeenSet() const { return m_attributesHasBeenSet; }
  template <typename AttributesT = Aws::Map<Aws::String, Aws::String>>
  void SetAttributes(AttributesT&& value) {
    m_attributesHasBeenSet = true;
    m_attributes = std::forward<AttributesT>(value);
  }
  template <typename AttributesT = Aws::Map<Aws::String, Aws::String>>
  StartTaskContactRequest& WithAttributes(AttributesT&& value) {
    SetAttributes(std::forward<AttributesT>(value));
    return *this;
  }
  template <typename AttributesKeyT = Aws::String, typename AttributesValueT = Aws::String>
  StartTaskContactRequest& AddAttributes(AttributesKeyT&& key, AttributesValueT&& value) {
    m_attributesHasBeenSet = true;
    m_attributes.emplace(std::forward<AttributesKeyT>(key), std::forward<AttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of a task that is shown to an agent in the Contact Control Panel
   * (CCP).</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  StartTaskContactRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A formatted URL that is shown to an agent in the Contact Control Panel (CCP).
   * Tasks can have the following reference types at the time of creation:
   * <code>URL</code> | <code>NUMBER</code> | <code>STRING</code> | <code>DATE</code>
   * | <code>EMAIL</code>. <code>ATTACHMENT</code> is not a supported reference type
   * during task creation.</p>
   */
  inline const Aws::Map<Aws::String, Reference>& GetReferences() const { return m_references; }
  inline bool ReferencesHasBeenSet() const { return m_referencesHasBeenSet; }
  template <typename ReferencesT = Aws::Map<Aws::String, Reference>>
  void SetReferences(ReferencesT&& value) {
    m_referencesHasBeenSet = true;
    m_references = std::forward<ReferencesT>(value);
  }
  template <typename ReferencesT = Aws::Map<Aws::String, Reference>>
  StartTaskContactRequest& WithReferences(ReferencesT&& value) {
    SetReferences(std::forward<ReferencesT>(value));
    return *this;
  }
  template <typename ReferencesKeyT = Aws::String, typename ReferencesValueT = Reference>
  StartTaskContactRequest& AddReferences(ReferencesKeyT&& key, ReferencesValueT&& value) {
    m_referencesHasBeenSet = true;
    m_references.emplace(std::forward<ReferencesKeyT>(key), std::forward<ReferencesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the task that is shown to an agent in the Contact Control
   * Panel (CCP).</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  StartTaskContactRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If not provided, the Amazon Web Services SDK
   * populates this field. For more information about idempotency, see <a
   * href="https://aws.amazon.com/builders-library/making-retries-safe-with-idempotent-APIs/">Making
   * retries safe with idempotent APIs</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartTaskContactRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp, in Unix Epoch seconds format, at which to start running the
   * inbound flow. The scheduled time cannot be in the past. It must be within up to
   * 6 days in future. </p>
   */
  inline const Aws::Utils::DateTime& GetScheduledTime() const { return m_scheduledTime; }
  inline bool ScheduledTimeHasBeenSet() const { return m_scheduledTimeHasBeenSet; }
  template <typename ScheduledTimeT = Aws::Utils::DateTime>
  void SetScheduledTime(ScheduledTimeT&& value) {
    m_scheduledTimeHasBeenSet = true;
    m_scheduledTime = std::forward<ScheduledTimeT>(value);
  }
  template <typename ScheduledTimeT = Aws::Utils::DateTime>
  StartTaskContactRequest& WithScheduledTime(ScheduledTimeT&& value) {
    SetScheduledTime(std::forward<ScheduledTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the task template. For more information about task
   * templates, see <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/task-templates.html">Create
   * task templates</a> in the <i>Amazon Connect Administrator Guide</i>. </p>
   */
  inline const Aws::String& GetTaskTemplateId() const { return m_taskTemplateId; }
  inline bool TaskTemplateIdHasBeenSet() const { return m_taskTemplateIdHasBeenSet; }
  template <typename TaskTemplateIdT = Aws::String>
  void SetTaskTemplateId(TaskTemplateIdT&& value) {
    m_taskTemplateIdHasBeenSet = true;
    m_taskTemplateId = std::forward<TaskTemplateIdT>(value);
  }
  template <typename TaskTemplateIdT = Aws::String>
  StartTaskContactRequest& WithTaskTemplateId(TaskTemplateIdT&& value) {
    SetTaskTemplateId(std::forward<TaskTemplateIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the quick connect. Tasks that are created by using
   * <code>QuickConnectId</code> will use the flow that is defined on agent or queue
   * quick connect. For more information about quick connects, see <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/quick-connects.html">Create
   * quick connects</a>.</p>
   */
  inline const Aws::String& GetQuickConnectId() const { return m_quickConnectId; }
  inline bool QuickConnectIdHasBeenSet() const { return m_quickConnectIdHasBeenSet; }
  template <typename QuickConnectIdT = Aws::String>
  void SetQuickConnectId(QuickConnectIdT&& value) {
    m_quickConnectIdHasBeenSet = true;
    m_quickConnectId = std::forward<QuickConnectIdT>(value);
  }
  template <typename QuickConnectIdT = Aws::String>
  StartTaskContactRequest& WithQuickConnectId(QuickConnectIdT&& value) {
    SetQuickConnectId(std::forward<QuickConnectIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The contactId that is <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/tasks.html#linked-tasks">related</a>
   * to this contact. Linking tasks together by using <code>RelatedContactID</code>
   * copies over contact attributes from the related task contact to the new task
   * contact. All updates to user-defined attributes in the new task contact are
   * limited to the individual contact ID, unlike what happens when tasks are linked
   * by using <code>PreviousContactID</code>. There are no limits to the number of
   * contacts that can be linked by using <code>RelatedContactId</code>. </p>
   */
  inline const Aws::String& GetRelatedContactId() const { return m_relatedContactId; }
  inline bool RelatedContactIdHasBeenSet() const { return m_relatedContactIdHasBeenSet; }
  template <typename RelatedContactIdT = Aws::String>
  void SetRelatedContactId(RelatedContactIdT&& value) {
    m_relatedContactIdHasBeenSet = true;
    m_relatedContactId = std::forward<RelatedContactIdT>(value);
  }
  template <typename RelatedContactIdT = Aws::String>
  StartTaskContactRequest& WithRelatedContactId(RelatedContactIdT&& value) {
    SetRelatedContactId(std::forward<RelatedContactIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of system defined key-value pairs stored on individual contact segments
   * (unique contact ID) using an attribute map. The attributes are standard Amazon
   * Connect attributes. They can be accessed in flows.</p> <p>Attribute keys can
   * include only alphanumeric, -, and _.</p> <p>This field can be used to set
   * Contact Expiry as a duration in minutes and set a UserId for the User who
   * created a task.</p>  <p>To set contact expiry, a ValueMap must be
   * specified containing the integer number of minutes the contact will be active
   * for before expiring, with <code>SegmentAttributes</code> like { <code>
   * "connect:ContactExpiry": {"ValueMap" : { "ExpiryDuration": { "ValueInteger":
   * 135}}}}</code>. </p> <p>To set the created by user, a valid AgentResourceId must
   * be supplied, with <code>SegmentAttributes</code> like {
   * <code>"connect:CreatedByUser" { "ValueString":
   * "arn:aws:connect:us-west-2:xxxxxxxxxxxx:instance/xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx/agent/xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx"}}}</code>.
   * </p>
   */
  inline const Aws::Map<Aws::String, SegmentAttributeValue>& GetSegmentAttributes() const { return m_segmentAttributes; }
  inline bool SegmentAttributesHasBeenSet() const { return m_segmentAttributesHasBeenSet; }
  template <typename SegmentAttributesT = Aws::Map<Aws::String, SegmentAttributeValue>>
  void SetSegmentAttributes(SegmentAttributesT&& value) {
    m_segmentAttributesHasBeenSet = true;
    m_segmentAttributes = std::forward<SegmentAttributesT>(value);
  }
  template <typename SegmentAttributesT = Aws::Map<Aws::String, SegmentAttributeValue>>
  StartTaskContactRequest& WithSegmentAttributes(SegmentAttributesT&& value) {
    SetSegmentAttributes(std::forward<SegmentAttributesT>(value));
    return *this;
  }
  template <typename SegmentAttributesKeyT = Aws::String, typename SegmentAttributesValueT = SegmentAttributeValue>
  StartTaskContactRequest& AddSegmentAttributes(SegmentAttributesKeyT&& key, SegmentAttributesValueT&& value) {
    m_segmentAttributesHasBeenSet = true;
    m_segmentAttributes.emplace(std::forward<SegmentAttributesKeyT>(key), std::forward<SegmentAttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of S3 presigned URLs of task attachments and their file name. You can
   * have a maximum of 5 attachments per task.</p>
   */
  inline const Aws::Vector<TaskAttachment>& GetAttachments() const { return m_attachments; }
  inline bool AttachmentsHasBeenSet() const { return m_attachmentsHasBeenSet; }
  template <typename AttachmentsT = Aws::Vector<TaskAttachment>>
  void SetAttachments(AttachmentsT&& value) {
    m_attachmentsHasBeenSet = true;
    m_attachments = std::forward<AttachmentsT>(value);
  }
  template <typename AttachmentsT = Aws::Vector<TaskAttachment>>
  StartTaskContactRequest& WithAttachments(AttachmentsT&& value) {
    SetAttachments(std::forward<AttachmentsT>(value));
    return *this;
  }
  template <typename AttachmentsT = TaskAttachment>
  StartTaskContactRequest& AddAttachments(AttachmentsT&& value) {
    m_attachmentsHasBeenSet = true;
    m_attachments.emplace_back(std::forward<AttachmentsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;

  Aws::String m_previousContactId;

  Aws::String m_contactFlowId;

  Aws::Map<Aws::String, Aws::String> m_attributes;

  Aws::String m_name;

  Aws::Map<Aws::String, Reference> m_references;

  Aws::String m_description;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Utils::DateTime m_scheduledTime{};

  Aws::String m_taskTemplateId;

  Aws::String m_quickConnectId;

  Aws::String m_relatedContactId;

  Aws::Map<Aws::String, SegmentAttributeValue> m_segmentAttributes;

  Aws::Vector<TaskAttachment> m_attachments;
  bool m_instanceIdHasBeenSet = false;
  bool m_previousContactIdHasBeenSet = false;
  bool m_contactFlowIdHasBeenSet = false;
  bool m_attributesHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_referencesHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_scheduledTimeHasBeenSet = false;
  bool m_taskTemplateIdHasBeenSet = false;
  bool m_quickConnectIdHasBeenSet = false;
  bool m_relatedContactIdHasBeenSet = false;
  bool m_segmentAttributesHasBeenSet = false;
  bool m_attachmentsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
