﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/AdditionalInferenceSpecificationDefinition.h>
#include <aws/sagemaker/model/DriftCheckBaselines.h>
#include <aws/sagemaker/model/InferenceSpecification.h>
#include <aws/sagemaker/model/MetadataProperties.h>
#include <aws/sagemaker/model/ModelApprovalStatus.h>
#include <aws/sagemaker/model/ModelLifeCycle.h>
#include <aws/sagemaker/model/ModelMetrics.h>
#include <aws/sagemaker/model/ModelPackageModelCard.h>
#include <aws/sagemaker/model/ModelPackageRegistrationType.h>
#include <aws/sagemaker/model/ModelPackageSecurityConfig.h>
#include <aws/sagemaker/model/ModelPackageStatus.h>
#include <aws/sagemaker/model/ModelPackageStatusDetails.h>
#include <aws/sagemaker/model/ModelPackageValidationSpecification.h>
#include <aws/sagemaker/model/SkipModelValidation.h>
#include <aws/sagemaker/model/SourceAlgorithmSpecification.h>
#include <aws/sagemaker/model/Tag.h>
#include <aws/sagemaker/model/UserContext.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SageMaker {
namespace Model {

/**
 * <p>A container for your trained model that can be deployed for SageMaker
 * inference. This can include inference code, artifacts, and metadata. The model
 * package type can be one of the following.</p> <ul> <li> <p>Versioned model: A
 * part of a model package group in Model Registry.</p> </li> <li> <p>Unversioned
 * model: Not part of a model package group and used in Amazon Web Services
 * Marketplace.</p> </li> </ul> <p>For more information, see <a
 * href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModelPackage.html">
 * <code>CreateModelPackage</code> </a>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sagemaker-2017-07-24/ModelPackage">AWS
 * API Reference</a></p>
 */
class ModelPackage {
 public:
  AWS_SAGEMAKER_API ModelPackage() = default;
  AWS_SAGEMAKER_API ModelPackage(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API ModelPackage& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SAGEMAKER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the model package. The name can be as follows:</p> <ul> <li>
   * <p>For a versioned model, the name is automatically generated by SageMaker Model
   * Registry and follows the format
   * '<code>ModelPackageGroupName/ModelPackageVersion</code>'.</p> </li> <li> <p>For
   * an unversioned model, you must provide the name.</p> </li> </ul>
   */
  inline const Aws::String& GetModelPackageName() const { return m_modelPackageName; }
  inline bool ModelPackageNameHasBeenSet() const { return m_modelPackageNameHasBeenSet; }
  template <typename ModelPackageNameT = Aws::String>
  void SetModelPackageName(ModelPackageNameT&& value) {
    m_modelPackageNameHasBeenSet = true;
    m_modelPackageName = std::forward<ModelPackageNameT>(value);
  }
  template <typename ModelPackageNameT = Aws::String>
  ModelPackage& WithModelPackageName(ModelPackageNameT&& value) {
    SetModelPackageName(std::forward<ModelPackageNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The model group to which the model belongs.</p>
   */
  inline const Aws::String& GetModelPackageGroupName() const { return m_modelPackageGroupName; }
  inline bool ModelPackageGroupNameHasBeenSet() const { return m_modelPackageGroupNameHasBeenSet; }
  template <typename ModelPackageGroupNameT = Aws::String>
  void SetModelPackageGroupName(ModelPackageGroupNameT&& value) {
    m_modelPackageGroupNameHasBeenSet = true;
    m_modelPackageGroupName = std::forward<ModelPackageGroupNameT>(value);
  }
  template <typename ModelPackageGroupNameT = Aws::String>
  ModelPackage& WithModelPackageGroupName(ModelPackageGroupNameT&& value) {
    SetModelPackageGroupName(std::forward<ModelPackageGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version number of a versioned model.</p>
   */
  inline int GetModelPackageVersion() const { return m_modelPackageVersion; }
  inline bool ModelPackageVersionHasBeenSet() const { return m_modelPackageVersionHasBeenSet; }
  inline void SetModelPackageVersion(int value) {
    m_modelPackageVersionHasBeenSet = true;
    m_modelPackageVersion = value;
  }
  inline ModelPackage& WithModelPackageVersion(int value) {
    SetModelPackageVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The package registration type of the model package. </p>
   */
  inline ModelPackageRegistrationType GetModelPackageRegistrationType() const { return m_modelPackageRegistrationType; }
  inline bool ModelPackageRegistrationTypeHasBeenSet() const { return m_modelPackageRegistrationTypeHasBeenSet; }
  inline void SetModelPackageRegistrationType(ModelPackageRegistrationType value) {
    m_modelPackageRegistrationTypeHasBeenSet = true;
    m_modelPackageRegistrationType = value;
  }
  inline ModelPackage& WithModelPackageRegistrationType(ModelPackageRegistrationType value) {
    SetModelPackageRegistrationType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the model package.</p>
   */
  inline const Aws::String& GetModelPackageArn() const { return m_modelPackageArn; }
  inline bool ModelPackageArnHasBeenSet() const { return m_modelPackageArnHasBeenSet; }
  template <typename ModelPackageArnT = Aws::String>
  void SetModelPackageArn(ModelPackageArnT&& value) {
    m_modelPackageArnHasBeenSet = true;
    m_modelPackageArn = std::forward<ModelPackageArnT>(value);
  }
  template <typename ModelPackageArnT = Aws::String>
  ModelPackage& WithModelPackageArn(ModelPackageArnT&& value) {
    SetModelPackageArn(std::forward<ModelPackageArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the model package.</p>
   */
  inline const Aws::String& GetModelPackageDescription() const { return m_modelPackageDescription; }
  inline bool ModelPackageDescriptionHasBeenSet() const { return m_modelPackageDescriptionHasBeenSet; }
  template <typename ModelPackageDescriptionT = Aws::String>
  void SetModelPackageDescription(ModelPackageDescriptionT&& value) {
    m_modelPackageDescriptionHasBeenSet = true;
    m_modelPackageDescription = std::forward<ModelPackageDescriptionT>(value);
  }
  template <typename ModelPackageDescriptionT = Aws::String>
  ModelPackage& WithModelPackageDescription(ModelPackageDescriptionT&& value) {
    SetModelPackageDescription(std::forward<ModelPackageDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time that the model package was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  ModelPackage& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Defines how to perform inference generation after a training job is run.</p>
   */
  inline const InferenceSpecification& GetInferenceSpecification() const { return m_inferenceSpecification; }
  inline bool InferenceSpecificationHasBeenSet() const { return m_inferenceSpecificationHasBeenSet; }
  template <typename InferenceSpecificationT = InferenceSpecification>
  void SetInferenceSpecification(InferenceSpecificationT&& value) {
    m_inferenceSpecificationHasBeenSet = true;
    m_inferenceSpecification = std::forward<InferenceSpecificationT>(value);
  }
  template <typename InferenceSpecificationT = InferenceSpecification>
  ModelPackage& WithInferenceSpecification(InferenceSpecificationT&& value) {
    SetInferenceSpecification(std::forward<InferenceSpecificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of algorithms that were used to create a model package.</p>
   */
  inline const SourceAlgorithmSpecification& GetSourceAlgorithmSpecification() const { return m_sourceAlgorithmSpecification; }
  inline bool SourceAlgorithmSpecificationHasBeenSet() const { return m_sourceAlgorithmSpecificationHasBeenSet; }
  template <typename SourceAlgorithmSpecificationT = SourceAlgorithmSpecification>
  void SetSourceAlgorithmSpecification(SourceAlgorithmSpecificationT&& value) {
    m_sourceAlgorithmSpecificationHasBeenSet = true;
    m_sourceAlgorithmSpecification = std::forward<SourceAlgorithmSpecificationT>(value);
  }
  template <typename SourceAlgorithmSpecificationT = SourceAlgorithmSpecification>
  ModelPackage& WithSourceAlgorithmSpecification(SourceAlgorithmSpecificationT&& value) {
    SetSourceAlgorithmSpecification(std::forward<SourceAlgorithmSpecificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies batch transform jobs that SageMaker runs to validate your model
   * package.</p>
   */
  inline const ModelPackageValidationSpecification& GetValidationSpecification() const { return m_validationSpecification; }
  inline bool ValidationSpecificationHasBeenSet() const { return m_validationSpecificationHasBeenSet; }
  template <typename ValidationSpecificationT = ModelPackageValidationSpecification>
  void SetValidationSpecification(ValidationSpecificationT&& value) {
    m_validationSpecificationHasBeenSet = true;
    m_validationSpecification = std::forward<ValidationSpecificationT>(value);
  }
  template <typename ValidationSpecificationT = ModelPackageValidationSpecification>
  ModelPackage& WithValidationSpecification(ValidationSpecificationT&& value) {
    SetValidationSpecification(std::forward<ValidationSpecificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the model package. This can be one of the following values.</p>
   * <ul> <li> <p> <code>PENDING</code> - The model package is pending being
   * created.</p> </li> <li> <p> <code>IN_PROGRESS</code> - The model package is in
   * the process of being created.</p> </li> <li> <p> <code>COMPLETED</code> - The
   * model package was successfully created.</p> </li> <li> <p> <code>FAILED</code> -
   * The model package failed.</p> </li> <li> <p> <code>DELETING</code> - The model
   * package is in the process of being deleted.</p> </li> </ul>
   */
  inline ModelPackageStatus GetModelPackageStatus() const { return m_modelPackageStatus; }
  inline bool ModelPackageStatusHasBeenSet() const { return m_modelPackageStatusHasBeenSet; }
  inline void SetModelPackageStatus(ModelPackageStatus value) {
    m_modelPackageStatusHasBeenSet = true;
    m_modelPackageStatus = value;
  }
  inline ModelPackage& WithModelPackageStatus(ModelPackageStatus value) {
    SetModelPackageStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the validation and image scan statuses of the model package.</p>
   */
  inline const ModelPackageStatusDetails& GetModelPackageStatusDetails() const { return m_modelPackageStatusDetails; }
  inline bool ModelPackageStatusDetailsHasBeenSet() const { return m_modelPackageStatusDetailsHasBeenSet; }
  template <typename ModelPackageStatusDetailsT = ModelPackageStatusDetails>
  void SetModelPackageStatusDetails(ModelPackageStatusDetailsT&& value) {
    m_modelPackageStatusDetailsHasBeenSet = true;
    m_modelPackageStatusDetails = std::forward<ModelPackageStatusDetailsT>(value);
  }
  template <typename ModelPackageStatusDetailsT = ModelPackageStatusDetails>
  ModelPackage& WithModelPackageStatusDetails(ModelPackageStatusDetailsT&& value) {
    SetModelPackageStatusDetails(std::forward<ModelPackageStatusDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether the model package is to be certified to be listed on Amazon Web
   * Services Marketplace. For information about listing model packages on Amazon Web
   * Services Marketplace, see <a
   * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-mkt-list.html">List
   * Your Algorithm or Model Package on Amazon Web Services Marketplace</a>.</p>
   */
  inline bool GetCertifyForMarketplace() const { return m_certifyForMarketplace; }
  inline bool CertifyForMarketplaceHasBeenSet() const { return m_certifyForMarketplaceHasBeenSet; }
  inline void SetCertifyForMarketplace(bool value) {
    m_certifyForMarketplaceHasBeenSet = true;
    m_certifyForMarketplace = value;
  }
  inline ModelPackage& WithCertifyForMarketplace(bool value) {
    SetCertifyForMarketplace(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The approval status of the model. This can be one of the following
   * values.</p> <ul> <li> <p> <code>APPROVED</code> - The model is approved</p>
   * </li> <li> <p> <code>REJECTED</code> - The model is rejected.</p> </li> <li> <p>
   * <code>PENDING_MANUAL_APPROVAL</code> - The model is waiting for manual
   * approval.</p> </li> </ul>
   */
  inline ModelApprovalStatus GetModelApprovalStatus() const { return m_modelApprovalStatus; }
  inline bool ModelApprovalStatusHasBeenSet() const { return m_modelApprovalStatusHasBeenSet; }
  inline void SetModelApprovalStatus(ModelApprovalStatus value) {
    m_modelApprovalStatusHasBeenSet = true;
    m_modelApprovalStatus = value;
  }
  inline ModelPackage& WithModelApprovalStatus(ModelApprovalStatus value) {
    SetModelApprovalStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the user who created or modified an experiment, trial,
   * trial component, lineage group, or project.</p>
   */
  inline const UserContext& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = UserContext>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = UserContext>
  ModelPackage& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata properties of the tracking entity, trial, or trial component.</p>
   */
  inline const MetadataProperties& GetMetadataProperties() const { return m_metadataProperties; }
  inline bool MetadataPropertiesHasBeenSet() const { return m_metadataPropertiesHasBeenSet; }
  template <typename MetadataPropertiesT = MetadataProperties>
  void SetMetadataProperties(MetadataPropertiesT&& value) {
    m_metadataPropertiesHasBeenSet = true;
    m_metadataProperties = std::forward<MetadataPropertiesT>(value);
  }
  template <typename MetadataPropertiesT = MetadataProperties>
  ModelPackage& WithMetadataProperties(MetadataPropertiesT&& value) {
    SetMetadataProperties(std::forward<MetadataPropertiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metrics for the model.</p>
   */
  inline const ModelMetrics& GetModelMetrics() const { return m_modelMetrics; }
  inline bool ModelMetricsHasBeenSet() const { return m_modelMetricsHasBeenSet; }
  template <typename ModelMetricsT = ModelMetrics>
  void SetModelMetrics(ModelMetricsT&& value) {
    m_modelMetricsHasBeenSet = true;
    m_modelMetrics = std::forward<ModelMetricsT>(value);
  }
  template <typename ModelMetricsT = ModelMetrics>
  ModelPackage& WithModelMetrics(ModelMetricsT&& value) {
    SetModelMetrics(std::forward<ModelMetricsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last time the model package was modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  inline bool LastModifiedTimeHasBeenSet() const { return m_lastModifiedTimeHasBeenSet; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  ModelPackage& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the user who created or modified an experiment, trial,
   * trial component, lineage group, or project.</p>
   */
  inline const UserContext& GetLastModifiedBy() const { return m_lastModifiedBy; }
  inline bool LastModifiedByHasBeenSet() const { return m_lastModifiedByHasBeenSet; }
  template <typename LastModifiedByT = UserContext>
  void SetLastModifiedBy(LastModifiedByT&& value) {
    m_lastModifiedByHasBeenSet = true;
    m_lastModifiedBy = std::forward<LastModifiedByT>(value);
  }
  template <typename LastModifiedByT = UserContext>
  ModelPackage& WithLastModifiedBy(LastModifiedByT&& value) {
    SetLastModifiedBy(std::forward<LastModifiedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description provided when the model approval is set.</p>
   */
  inline const Aws::String& GetApprovalDescription() const { return m_approvalDescription; }
  inline bool ApprovalDescriptionHasBeenSet() const { return m_approvalDescriptionHasBeenSet; }
  template <typename ApprovalDescriptionT = Aws::String>
  void SetApprovalDescription(ApprovalDescriptionT&& value) {
    m_approvalDescriptionHasBeenSet = true;
    m_approvalDescription = std::forward<ApprovalDescriptionT>(value);
  }
  template <typename ApprovalDescriptionT = Aws::String>
  ModelPackage& WithApprovalDescription(ApprovalDescriptionT&& value) {
    SetApprovalDescription(std::forward<ApprovalDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The machine learning domain of your model package and its components. Common
   * machine learning domains include computer vision and natural language
   * processing.</p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  ModelPackage& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The machine learning task your model package accomplishes. Common machine
   * learning tasks include object detection and image classification.</p>
   */
  inline const Aws::String& GetTask() const { return m_task; }
  inline bool TaskHasBeenSet() const { return m_taskHasBeenSet; }
  template <typename TaskT = Aws::String>
  void SetTask(TaskT&& value) {
    m_taskHasBeenSet = true;
    m_task = std::forward<TaskT>(value);
  }
  template <typename TaskT = Aws::String>
  ModelPackage& WithTask(TaskT&& value) {
    SetTask(std::forward<TaskT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Simple Storage Service path where the sample payload are stored.
   * This path must point to a single gzip compressed tar archive (.tar.gz
   * suffix).</p>
   */
  inline const Aws::String& GetSamplePayloadUrl() const { return m_samplePayloadUrl; }
  inline bool SamplePayloadUrlHasBeenSet() const { return m_samplePayloadUrlHasBeenSet; }
  template <typename SamplePayloadUrlT = Aws::String>
  void SetSamplePayloadUrl(SamplePayloadUrlT&& value) {
    m_samplePayloadUrlHasBeenSet = true;
    m_samplePayloadUrl = std::forward<SamplePayloadUrlT>(value);
  }
  template <typename SamplePayloadUrlT = Aws::String>
  ModelPackage& WithSamplePayloadUrl(SamplePayloadUrlT&& value) {
    SetSamplePayloadUrl(std::forward<SamplePayloadUrlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of additional Inference Specification objects.</p>
   */
  inline const Aws::Vector<AdditionalInferenceSpecificationDefinition>& GetAdditionalInferenceSpecifications() const {
    return m_additionalInferenceSpecifications;
  }
  inline bool AdditionalInferenceSpecificationsHasBeenSet() const { return m_additionalInferenceSpecificationsHasBeenSet; }
  template <typename AdditionalInferenceSpecificationsT = Aws::Vector<AdditionalInferenceSpecificationDefinition>>
  void SetAdditionalInferenceSpecifications(AdditionalInferenceSpecificationsT&& value) {
    m_additionalInferenceSpecificationsHasBeenSet = true;
    m_additionalInferenceSpecifications = std::forward<AdditionalInferenceSpecificationsT>(value);
  }
  template <typename AdditionalInferenceSpecificationsT = Aws::Vector<AdditionalInferenceSpecificationDefinition>>
  ModelPackage& WithAdditionalInferenceSpecifications(AdditionalInferenceSpecificationsT&& value) {
    SetAdditionalInferenceSpecifications(std::forward<AdditionalInferenceSpecificationsT>(value));
    return *this;
  }
  template <typename AdditionalInferenceSpecificationsT = AdditionalInferenceSpecificationDefinition>
  ModelPackage& AddAdditionalInferenceSpecifications(AdditionalInferenceSpecificationsT&& value) {
    m_additionalInferenceSpecificationsHasBeenSet = true;
    m_additionalInferenceSpecifications.emplace_back(std::forward<AdditionalInferenceSpecificationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The URI of the source for the model package.</p>
   */
  inline const Aws::String& GetSourceUri() const { return m_sourceUri; }
  inline bool SourceUriHasBeenSet() const { return m_sourceUriHasBeenSet; }
  template <typename SourceUriT = Aws::String>
  void SetSourceUri(SourceUriT&& value) {
    m_sourceUriHasBeenSet = true;
    m_sourceUri = std::forward<SourceUriT>(value);
  }
  template <typename SourceUriT = Aws::String>
  ModelPackage& WithSourceUri(SourceUriT&& value) {
    SetSourceUri(std::forward<SourceUriT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ModelPackageSecurityConfig& GetSecurityConfig() const { return m_securityConfig; }
  inline bool SecurityConfigHasBeenSet() const { return m_securityConfigHasBeenSet; }
  template <typename SecurityConfigT = ModelPackageSecurityConfig>
  void SetSecurityConfig(SecurityConfigT&& value) {
    m_securityConfigHasBeenSet = true;
    m_securityConfig = std::forward<SecurityConfigT>(value);
  }
  template <typename SecurityConfigT = ModelPackageSecurityConfig>
  ModelPackage& WithSecurityConfig(SecurityConfigT&& value) {
    SetSecurityConfig(std::forward<SecurityConfigT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ModelPackageModelCard& GetModelCard() const { return m_modelCard; }
  inline bool ModelCardHasBeenSet() const { return m_modelCardHasBeenSet; }
  template <typename ModelCardT = ModelPackageModelCard>
  void SetModelCard(ModelCardT&& value) {
    m_modelCardHasBeenSet = true;
    m_modelCard = std::forward<ModelCardT>(value);
  }
  template <typename ModelCardT = ModelPackageModelCard>
  ModelPackage& WithModelCard(ModelCardT&& value) {
    SetModelCard(std::forward<ModelCardT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A structure describing the current state of the model in its life cycle.
   * </p>
   */
  inline const ModelLifeCycle& GetModelLifeCycle() const { return m_modelLifeCycle; }
  inline bool ModelLifeCycleHasBeenSet() const { return m_modelLifeCycleHasBeenSet; }
  template <typename ModelLifeCycleT = ModelLifeCycle>
  void SetModelLifeCycle(ModelLifeCycleT&& value) {
    m_modelLifeCycleHasBeenSet = true;
    m_modelLifeCycle = std::forward<ModelLifeCycleT>(value);
  }
  template <typename ModelLifeCycleT = ModelLifeCycle>
  ModelPackage& WithModelLifeCycle(ModelLifeCycleT&& value) {
    SetModelLifeCycle(std::forward<ModelLifeCycleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of the tags associated with the model package. For more information,
   * see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging
   * Amazon Web Services resources</a> in the <i>Amazon Web Services General
   * Reference Guide</i>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  ModelPackage& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  ModelPackage& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metadata properties for the model package. </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetCustomerMetadataProperties() const { return m_customerMetadataProperties; }
  inline bool CustomerMetadataPropertiesHasBeenSet() const { return m_customerMetadataPropertiesHasBeenSet; }
  template <typename CustomerMetadataPropertiesT = Aws::Map<Aws::String, Aws::String>>
  void SetCustomerMetadataProperties(CustomerMetadataPropertiesT&& value) {
    m_customerMetadataPropertiesHasBeenSet = true;
    m_customerMetadataProperties = std::forward<CustomerMetadataPropertiesT>(value);
  }
  template <typename CustomerMetadataPropertiesT = Aws::Map<Aws::String, Aws::String>>
  ModelPackage& WithCustomerMetadataProperties(CustomerMetadataPropertiesT&& value) {
    SetCustomerMetadataProperties(std::forward<CustomerMetadataPropertiesT>(value));
    return *this;
  }
  template <typename CustomerMetadataPropertiesKeyT = Aws::String, typename CustomerMetadataPropertiesValueT = Aws::String>
  ModelPackage& AddCustomerMetadataProperties(CustomerMetadataPropertiesKeyT&& key, CustomerMetadataPropertiesValueT&& value) {
    m_customerMetadataPropertiesHasBeenSet = true;
    m_customerMetadataProperties.emplace(std::forward<CustomerMetadataPropertiesKeyT>(key),
                                         std::forward<CustomerMetadataPropertiesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Represents the drift check baselines that can be used when the model monitor
   * is set using the model package.</p>
   */
  inline const DriftCheckBaselines& GetDriftCheckBaselines() const { return m_driftCheckBaselines; }
  inline bool DriftCheckBaselinesHasBeenSet() const { return m_driftCheckBaselinesHasBeenSet; }
  template <typename DriftCheckBaselinesT = DriftCheckBaselines>
  void SetDriftCheckBaselines(DriftCheckBaselinesT&& value) {
    m_driftCheckBaselinesHasBeenSet = true;
    m_driftCheckBaselines = std::forward<DriftCheckBaselinesT>(value);
  }
  template <typename DriftCheckBaselinesT = DriftCheckBaselines>
  ModelPackage& WithDriftCheckBaselines(DriftCheckBaselinesT&& value) {
    SetDriftCheckBaselines(std::forward<DriftCheckBaselinesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates if you want to skip model validation.</p>
   */
  inline SkipModelValidation GetSkipModelValidation() const { return m_skipModelValidation; }
  inline bool SkipModelValidationHasBeenSet() const { return m_skipModelValidationHasBeenSet; }
  inline void SetSkipModelValidation(SkipModelValidation value) {
    m_skipModelValidationHasBeenSet = true;
    m_skipModelValidation = value;
  }
  inline ModelPackage& WithSkipModelValidation(SkipModelValidation value) {
    SetSkipModelValidation(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_modelPackageName;

  Aws::String m_modelPackageGroupName;

  int m_modelPackageVersion{0};

  ModelPackageRegistrationType m_modelPackageRegistrationType{ModelPackageRegistrationType::NOT_SET};

  Aws::String m_modelPackageArn;

  Aws::String m_modelPackageDescription;

  Aws::Utils::DateTime m_creationTime{};

  InferenceSpecification m_inferenceSpecification;

  SourceAlgorithmSpecification m_sourceAlgorithmSpecification;

  ModelPackageValidationSpecification m_validationSpecification;

  ModelPackageStatus m_modelPackageStatus{ModelPackageStatus::NOT_SET};

  ModelPackageStatusDetails m_modelPackageStatusDetails;

  bool m_certifyForMarketplace{false};

  ModelApprovalStatus m_modelApprovalStatus{ModelApprovalStatus::NOT_SET};

  UserContext m_createdBy;

  MetadataProperties m_metadataProperties;

  ModelMetrics m_modelMetrics;

  Aws::Utils::DateTime m_lastModifiedTime{};

  UserContext m_lastModifiedBy;

  Aws::String m_approvalDescription;

  Aws::String m_domain;

  Aws::String m_task;

  Aws::String m_samplePayloadUrl;

  Aws::Vector<AdditionalInferenceSpecificationDefinition> m_additionalInferenceSpecifications;

  Aws::String m_sourceUri;

  ModelPackageSecurityConfig m_securityConfig;

  ModelPackageModelCard m_modelCard;

  ModelLifeCycle m_modelLifeCycle;

  Aws::Vector<Tag> m_tags;

  Aws::Map<Aws::String, Aws::String> m_customerMetadataProperties;

  DriftCheckBaselines m_driftCheckBaselines;

  SkipModelValidation m_skipModelValidation{SkipModelValidation::NOT_SET};
  bool m_modelPackageNameHasBeenSet = false;
  bool m_modelPackageGroupNameHasBeenSet = false;
  bool m_modelPackageVersionHasBeenSet = false;
  bool m_modelPackageRegistrationTypeHasBeenSet = false;
  bool m_modelPackageArnHasBeenSet = false;
  bool m_modelPackageDescriptionHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_inferenceSpecificationHasBeenSet = false;
  bool m_sourceAlgorithmSpecificationHasBeenSet = false;
  bool m_validationSpecificationHasBeenSet = false;
  bool m_modelPackageStatusHasBeenSet = false;
  bool m_modelPackageStatusDetailsHasBeenSet = false;
  bool m_certifyForMarketplaceHasBeenSet = false;
  bool m_modelApprovalStatusHasBeenSet = false;
  bool m_createdByHasBeenSet = false;
  bool m_metadataPropertiesHasBeenSet = false;
  bool m_modelMetricsHasBeenSet = false;
  bool m_lastModifiedTimeHasBeenSet = false;
  bool m_lastModifiedByHasBeenSet = false;
  bool m_approvalDescriptionHasBeenSet = false;
  bool m_domainHasBeenSet = false;
  bool m_taskHasBeenSet = false;
  bool m_samplePayloadUrlHasBeenSet = false;
  bool m_additionalInferenceSpecificationsHasBeenSet = false;
  bool m_sourceUriHasBeenSet = false;
  bool m_securityConfigHasBeenSet = false;
  bool m_modelCardHasBeenSet = false;
  bool m_modelLifeCycleHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_customerMetadataPropertiesHasBeenSet = false;
  bool m_driftCheckBaselinesHasBeenSet = false;
  bool m_skipModelValidationHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
