﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/workmail/WorkMailRequest.h>
#include <aws/workmail/WorkMail_EXPORTS.h>

#include <utility>

namespace Aws {
namespace WorkMail {
namespace Model {

/**
 */
class ListMobileDeviceAccessOverridesRequest : public WorkMailRequest {
 public:
  AWS_WORKMAIL_API ListMobileDeviceAccessOverridesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListMobileDeviceAccessOverrides"; }

  AWS_WORKMAIL_API Aws::String SerializePayload() const override;

  AWS_WORKMAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The WorkMail organization under which to list mobile device access
   * overrides.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  ListMobileDeviceAccessOverridesRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The WorkMail user under which you list the mobile device access overrides.
   * Accepts the following types of user identities:</p> <ul> <li> <p>User ID:
   * <code>12345678-1234-1234-1234-123456789012</code> or
   * <code>S-1-1-12-1234567890-123456789-123456789-1234</code> </p> </li> <li>
   * <p>Email address: <code>user@domain.tld</code> </p> </li> <li> <p>User name:
   * <code>user</code> </p> </li> </ul>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  ListMobileDeviceAccessOverridesRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The mobile device to which the access override applies.</p>
   */
  inline const Aws::String& GetDeviceId() const { return m_deviceId; }
  inline bool DeviceIdHasBeenSet() const { return m_deviceIdHasBeenSet; }
  template <typename DeviceIdT = Aws::String>
  void SetDeviceId(DeviceIdT&& value) {
    m_deviceIdHasBeenSet = true;
    m_deviceId = std::forward<DeviceIdT>(value);
  }
  template <typename DeviceIdT = Aws::String>
  ListMobileDeviceAccessOverridesRequest& WithDeviceId(DeviceIdT&& value) {
    SetDeviceId(std::forward<DeviceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to use to retrieve the next page of results. The first call does
   * not require a token.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListMobileDeviceAccessOverridesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in a single call.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListMobileDeviceAccessOverridesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_organizationId;

  Aws::String m_userId;

  Aws::String m_deviceId;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_organizationIdHasBeenSet = false;
  bool m_userIdHasBeenSet = false;
  bool m_deviceIdHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkMail
}  // namespace Aws
